<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");







function debug_log($message) {
    $log_file = __DIR__ . '/asaas_cashout_log.txt';
    $timestamp = date('Y-m-d H:i:s');
    
    // Verifica se o message é um array ou objeto e converte para string
    if (is_array($message) || is_object($message)) {
        $message = print_r($message, true); // ou var_export($message, true)
    }
    
    file_put_contents($log_file, "[$timestamp] $message\n", FILE_APPEND);
}



















function generateRandomString($length = 32) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[random_int(0, $charLength - 1)];
    }
    return $randomString;
}

// Função para validar CPF
function validarCPF($cpf) {
    // Remove caracteres não numéricos
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    
    // Verifica se tem 11 dígitos
    if (strlen($cpf) != 11) {
        return false;
    }
    
    // Verifica se todos os dígitos são iguais
    if (preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    
    // Calcula os dígitos verificadores
    for ($t = 9; $t < 11; $t++) {
        for ($d = 0, $c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) {
            return false;
        }
    }
    return true;
}

// Função para validar CNPJ
function validarCNPJ($cnpj) {
    // Remove caracteres não numéricos
    $cnpj = preg_replace('/[^0-9]/', '', $cnpj);
    
    // Verifica se tem 14 dígitos
    if (strlen($cnpj) != 14) {
        return false;
    }
    
    // Verifica se todos os dígitos são iguais
    if (preg_match('/(\d)\1{13}/', $cnpj)) {
        return false;
    }
    
    // Calcula os dígitos verificadores
    for ($i = 0, $j = 5, $soma = 0; $i < 12; $i++) {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }
    
    $resto = $soma % 11;
    if ($cnpj[12] != ($resto < 2 ? 0 : 11 - $resto)) {
        return false;
    }
    
    for ($i = 0, $j = 6, $soma = 0; $i < 13; $i++) {
        $soma += $cnpj[$i] * $j;
        $j = ($j == 2) ? 9 : $j - 1;
    }
    
    $resto = $soma % 11;
    return $cnpj[13] == ($resto < 2 ? 0 : 11 - $resto);
}

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $requestData = json_decode(file_get_contents('php://input'), true);

    if (!isset($requestData['api-key'], $requestData['name'], $requestData['keypix'], $requestData['amount'])) {
        http_response_code(400);
        echo json_encode(["error" => "Falha na solicitação: Dados incompletos"]);
        exit;
    }

    // Valida CPF/CNPJ apenas se fornecido
    $cpfLimpo = '';
    $isCPF = false;
    $isCNPJ = false;
    
    if (isset($requestData['cpf']) && !empty($requestData['cpf'])) {
        $cpfLimpo = preg_replace('/\D/', '', $requestData['cpf']);
        $isCPF = strlen($cpfLimpo) == 11;
        $isCNPJ = strlen($cpfLimpo) == 14;

        if ($isCPF && !validarCPF($cpfLimpo)) {
            echo json_encode(["status" => "error", "message" => "CPF inválido"]);
            exit;
        }

        if ($isCNPJ && !validarCNPJ($cpfLimpo)) {
            echo json_encode(["status" => "error", "message" => "CNPJ inválido"]);
            exit;
        }

        if (!$isCPF && !$isCNPJ) {
            echo json_encode(["status" => "error", "message" => "CPF/CNPJ deve ter 11 ou 14 dígitos"]);
            exit;
        }
    }

    include '../../conectarbanco.php';

    $conn = new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);
    if ($conn->connect_error) {
        die("Falha na conexão com o banco de dados: " . $conn->connect_error);
    }

    $sql = "SELECT user_id FROM users_key WHERE api_key = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $requestData['api-key']);
    $stmt->execute();
    $stmt->store_result();

    if ($stmt->num_rows > 0) {
        $stmt->bind_result($usuario);
        $stmt->fetch();
        $stmt->close();

        if (empty($usuario)) {
            echo json_encode(["status" => "error", "message" => "user_id não encontrado"]);
            exit;
        }

        $sql_saldo = "SELECT saldo FROM users WHERE user_id = ?";
        $stmt_saldo = $conn->prepare($sql_saldo);
        $stmt_saldo->bind_param("s", $usuario);
        $stmt_saldo->execute();
        $stmt_saldo->bind_result($saldo);
        $stmt_saldo->fetch();
        $stmt_saldo->close();

        $amount = floatval($requestData['amount']);
        if ($saldo < $amount) {
            echo json_encode(["status" => "error", "message" => "Saldo insuficiente"]);
            exit;
        }

        $adquirente_ref = '';
        $sql_adquirente = "SELECT adquirente FROM adquirentes WHERE status = 1 LIMIT 1";
        $stmt_adquirente = $conn->prepare($sql_adquirente);
        $stmt_adquirente->execute();
        $stmt_adquirente->bind_result($adquirente_ref);
        $stmt_adquirente->fetch();
        $stmt_adquirente->close();

        if (empty($adquirente_ref)) {
            $adquirente_ref = 'default';
        }

        $apiUrl = '';
        
        if ($adquirente_ref === 'suitpay') {
            $apiUrl = "https://api.getpaybr.com/c1/adquirente/suitpay/";
        } elseif ($adquirente_ref === 'bspay') {
            $apiUrl = "https://api.getpaybr.com/c1/adquirente/bspay/";
        } elseif ($adquirente_ref === 'pagpix') {
            $apiUrl = "https://api.getpaybr.com/c1/adquirente/pagpix/";
        } elseif ($adquirente_ref === 'asaas') {
            $apiUrl = "https://api.getpaybr.com/c1/adquirente/asaas/";
        } else {
            echo json_encode(["status" => "error", "message" => "Adquirente não reconhecido"]);
            exit;
        }

        $horaAtual = date('Y-m-d H:i:s');

        $sql_taxa = "SELECT taxa_cash_out FROM users WHERE user_id = ?";
        $stmt_taxa = $conn->prepare($sql_taxa);
        $stmt_taxa->bind_param("s", $usuario);
        $stmt_taxa->execute();
        $stmt_taxa->bind_result($taxa_cash_out);
        $stmt_taxa->fetch();
        $stmt_taxa->close();

        if ($taxa_cash_out === null) {
            $taxa_cash_out = 4; 
        }

        $cash_out_liquido = $amount - $taxa_cash_out;

        // Limpa chave PIX
        $keypixLimpo = preg_replace('/\D/', '', $requestData['keypix']);

        // Detecta o tipo da chave PIX
        if (filter_var($requestData['keypix'], FILTER_VALIDATE_EMAIL)) {
            $keyType = "EMAIL";
        } elseif (preg_match('/^\+?\d{10,15}$/', $requestData['keypix'])) {
            $keyType = "PHONE";
        } elseif (preg_match('/^\d{11}$/', $keypixLimpo)) {
            $keyType = "CPF";
        } elseif (preg_match('/^\d{14}$/', $keypixLimpo)) {
            $keyType = "CNPJ";
        } else {
            $keyType = "RANDOM_KEY";
        }

        // Garante que o nome é completo (mínimo 2 palavras)
        $nomeCompleto = trim($requestData['name']);
        if (str_word_count($nomeCompleto) < 2) {
            echo json_encode(["status" => "error", "message" => "O nome do favorecido deve ser completo."]);
            exit;
        }

        // Monta o array conforme o padrão da Asaas se for adquirente asaas
        if ($adquirente_ref === 'asaas') {
            $apiData = [
                "beneficiaryName" => $nomeCompleto,
                "amount"          => $cash_out_liquido,
                "keypix"          => $requestData['keypix'],
                // Enviar o tipo exatamente como a Asaas espera (EMAIL, CPF, CNPJ, PHONE, RANDOM_KEY)
                "keyType"         => $keyType
            ];
            
            // Adiciona CPF apenas se fornecido
            if (!empty($cpfLimpo)) {
                $apiData["cpf"] = $cpfLimpo;
            }
        } else {
            // Mantém o padrão antigo para outros adquirentes
            $apiData = [
                "keyseguranca"    => "24142414dertinhoiakdsj4847jdks92m",
                "amount"          => $cash_out_liquido,
                "beneficiaryName" => $nomeCompleto,
                "keypix"          => $requestData['keypix'],
                "keyType"         => strtolower($keyType)
            ];
            
            // Adiciona CPF apenas se fornecido
            if (!empty($cpfLimpo)) {
                $apiData["cpf"] = $cpfLimpo;
            }
        }

        $apiOptions = [
            'http' => [
                'method'  => 'POST',
                'header'  => 'Content-Type: application/json',
                'content' => json_encode($apiData)
            ]
        ];

       $ch = curl_init($apiUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($apiData));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
$apiResponse = curl_exec($ch);

if (curl_errno($ch)) {
    echo json_encode([
        "status" => "error",
        "message" => "Erro cURL: " . curl_error($ch)
    ]);
    curl_close($ch);
    exit;
}

$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);


     
        // // echo "\n[DEBUG] URL chamada: $apiUrl\n";
        // echo "\n[DEBUG] Resposta da API:\n";
        // var_dump($apiResponse);

        // echo "</pre>";
        // die;

        if ($apiResponse === false) {
            echo json_encode(["status" => "error", "message" => "Erro K-101"]);
        } else {
            $apiResponseData = json_decode($apiResponse, true);

            // Log para debug
            error_log("API Response: " . $apiResponse);
            error_log("API Response Data: " . print_r($apiResponseData, true));

            if (isset($apiResponseData['status']) && $apiResponseData['status'] === 200) {
                $asaasInfo = isset($apiResponseData['asaas']) && is_array($apiResponseData['asaas']) ? $apiResponseData['asaas'] : null;
                echo json_encode(["status" => "success", "message" => "K-200", "asaas" => $asaasInfo]);

                $externalReference = generateRandomString(32);
                $status = '2';
                $pixKey = $requestData['keypix'];
                $type = 'PIX';
                // Guarda o ID de transferência retornado pelo Asaas (quando adquirente é asaas)
                $idtransaction = '';
                if (is_array($asaasInfo) && isset($asaasInfo['id'])) {
                    $idtransaction = (string)$asaasInfo['id'];
                }

                $sql_insert = "INSERT INTO solicitacoes_cash_out 
                    (user_id, externalreference, amount, beneficiaryname, beneficiarydocument, pix, type, pixkey, date, status, idtransaction, taxa_cash_out, cash_out_liquido) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

                if ($stmt_insert = $conn->prepare($sql_insert)) {
                    $stmt_insert->bind_param(
                        "sssssssssssdd",
                        $usuario,
                        $externalReference,
                        $requestData['amount'],
                        $requestData['name'],
                        $requestData['cpf'],
                        $pixKey,
                        $type,
                        $pixKey,
                        $horaAtual,
                        $status,
                        $idtransaction,
                        $taxa_cash_out,
                        $cash_out_liquido
                    );

                    $stmt_insert->execute();
                    $stmt_insert->close();

                    $finalApiResponse = file_get_contents("https://api.getpaybr.com/cron/executor1.php");
                } else {
                    error_log("Erro ao preparar a consulta SQL para inserção");
                }
            } else {
                // Construir mensagem amigável para o admin
                $errorMessage = "Erro na API do adquirente";
                if (isset($apiResponseData['asaas']['errors']) && is_array($apiResponseData['asaas']['errors'])) {
                    $descriptions = [];
                    $firstCode = null;
                    foreach ($apiResponseData['asaas']['errors'] as $err) {
                        if ($firstCode === null && isset($err['code'])) {
                            $firstCode = $err['code'];
                        }
                        $descriptions[] = $err['description'] ?? $err['code'] ?? json_encode($err);
                    }
                    $errorMessage = implode('; ', $descriptions);
                    echo json_encode(["status" => "error", "message" => $errorMessage, "code" => $firstCode]);
                } else {
                    if (isset($apiResponseData['message'])) {
                        $errorMessage = is_array($apiResponseData['message']) ? json_encode($apiResponseData['message']) : $apiResponseData['message'];
                    }
                    echo json_encode(["status" => "error", "message" => $errorMessage]);
                }
            }
        }
    } else {
        http_response_code(401);
        echo json_encode(["error" => "Falha na autenticação - K-402"]);
    }

    $conn->close();
} else {
    http_response_code(405);
    echo json_encode(["error" => "Método não permitido"]);
}
?>
