# 📚 Documentação da API de Pagamentos

## 🔗 Endpoint Base
```
https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php
```

---

## 📋 Visão Geral

Esta API permite processar pagamentos e consultar o status de transações através de dois métodos HTTP:

- **POST**: Processar novos pagamentos
- **GET**: Consultar status de transações existentes

---

## 🔑 Autenticação

Todas as requisições requerem uma **API Key** válida e ativa.

### Como obter a API Key:
1. Acesse o painel administrativo
2. Vá para a seção de usuários
3. Gere ou copie sua API Key

---

## 💳 1. PROCESSAR PAGAMENTO (POST)

### Endpoint
```
POST /processa_pagamento_api.php
```

### Headers
```
Content-Type: application/json
```

### Parâmetros Obrigatórios
| Campo | Tipo | Descrição | Exemplo |
|-------|------|-----------|---------|
| `api-key` | string | Sua chave de API | `"7115d6b39791cca550f4e3cc"` |
| `name` | string | Nome completo do cliente | `"João Silva"` |
| `cpf` | string | CPF/CNPJ do cliente | `"123.456.789-00"` |
| `amount` | number | Valor do pagamento | `50.00` |
| `tipoPagamento` | string | Tipo de pagamento | `"PIX"`, `"BOLETO"`, `"CREDIT_CARD"` |
| `email` | string | Email do cliente | `"joao@email.com"` |

### Parâmetros Opcionais
| Campo | Tipo | Descrição | Exemplo |
|-------|------|-----------|---------|
| `telefone` | string | Telefone do cliente | `"(11) 99999-9999"` |
| `user_id` | string | ID do usuário (padrão: usuário da API key) | `"123"` |

### Parâmetros para Cartão de Crédito
| Campo | Tipo | Descrição | Exemplo |
|-------|------|-----------|---------|
| `creditCard` | object | Dados do cartão | Ver exemplo abaixo |
| `creditCardHolderInfo` | object | Dados do titular | Ver exemplo abaixo |

### Exemplo de Requisição - PIX
```json
{
  "api-key": "7115d6b39791cca550f4e3cc",
  "name": "João Silva Teste",
  "cpf": "123.456.789-00",
  "amount": 50.00,
  "tipoPagamento": "PIX",
  "email": "joao@teste.com",
  "telefone": "(11) 99999-9999"
}
```

### Exemplo de Requisição - Cartão de Crédito
```json
{
  "api-key": "7115d6b39791cca550f4e3cc",
  "name": "João Silva Teste",
  "cpf": "123.456.789-00",
  "amount": 50.00,
  "tipoPagamento": "CREDIT_CARD",
  "email": "joao@teste.com",
  "creditCard": {
    "holderName": "JOAO SILVA TESTE",
    "number": "4111111111111111",
    "expiryMonth": "12",
    "expiryYear": "2025",
    "ccv": "123"
  },
  "creditCardHolderInfo": {
    "name": "João Silva Teste",
    "email": "joao@teste.com",
    "cpfCnpj": "12345678900",
    "postalCode": "01234-567",
    "addressNumber": "123",
    "addressComplement": "Apto 1",
    "phone": "11999999999"
  }
}
```

### Exemplo de Requisição - Boleto
```json
{
  "api-key": "7115d6b39791cca550f4e3cc",
  "name": "João Silva Teste",
  "cpf": "123.456.789-00",
  "amount": 50.00,
  "tipoPagamento": "BOLETO",
  "email": "joao@teste.com"
}
```

### Resposta de Sucesso - PIX
```json
{
  "status": "success",
  "idTransaction": "pay_8y3fyzdkeros3307",
  "tipo": "PIX",
  "paymentCode": "00020126580014br.gov.bcb.pix0136123e4567-e89b-12d3-a456-426614174000520400005303986540510.005802BR5913João Silva6008São Paulo62070503***6304E2CA",
  "qrCodeImage": "data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAA...",
  "asaas": {
    "id": "pay_8y3fyzdkeros3307",
    "status": "PENDING",
    "value": 50.00,
    "dueDate": "2025-08-30"
  }
}
```

### Resposta de Sucesso - Cartão
```json
{
  "status": "success",
  "idTransaction": "pay_8y3fyzdkeros3308",
  "tipo": "CREDIT_CARD",
  "asaas": {
    "id": "pay_8y3fyzdkeros3308",
    "status": "CONFIRMED",
    "value": 50.00,
    "dueDate": "2025-08-30"
  }
}
```

### Resposta de Sucesso - Boleto
```json
{
  "status": "success",
  "idTransaction": "pay_8y3fyzdkeros3309",
  "tipo": "BOLETO",
  "boletoUrl": "https://www.asaas.com/boleto/123456",
  "asaas": {
    "id": "pay_8y3fyzdkeros3309",
    "status": "PENDING",
    "value": 50.00,
    "dueDate": "2025-08-30"
  }
}
```

---

## 🔍 2. CONSULTAR STATUS (GET)

### Endpoint
```
GET /processa_pagamento_api.php?api-key=SUA_API_KEY&idTransaction=ID_TRANSACAO
```

### Parâmetros
| Campo | Tipo | Descrição | Exemplo |
|-------|------|-----------|---------|
| `api-key` | string | Sua chave de API | `7115d6b39791cca550f4e3cc` |
| `idTransaction` | string | ID da transação | `pay_8y3fyzdkeros3307` |

### Exemplo de URL
```
https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php?api-key=7115d6b39791cca550f4e3cc&idTransaction=pay_8y3fyzdkeros3307
```

### Resposta de Sucesso
```json
{
  "status": "success",
  "pedido": {
    "idTransaction": "pay_8y3fyzdkeros3307",
    "externalReference": "8600b4e655f342c49efb06b9569817e6",
    "amount": "50.00",
    "clientName": "João Silva Teste",
    "clientDocument": "39448004869",
    "clientEmail": "joao@teste.com",
    "realData": "2025-08-30",
    "status": "PENDING",
    "tipoTransacao": "PIX",
    "descTransacao": "Pagamento via Pix",
    "depositoLiquido": "50",
    "adquirenteRef": "asaas"
  },
  "asaas": {
    "status": "PENDING",
    "dueDate": "2025-08-30",
    "paymentDate": null,
    "value": 50
  }
}
```

---

## 📊 Status das Transações

### Status Possíveis
| Status | Descrição |
|--------|-----------|
| `PENDING` | Aguardando pagamento |
| `CONFIRMED` | Pagamento confirmado |
| `OVERDUE` | Pagamento vencido |
| `REFUNDED` | Pagamento estornado |
| `CANCELLED` | Pagamento cancelado |
| `WAITING_FOR_APPROVAL` | Aguardando aprovação |

---

## 🚨 Códigos de Erro

### 400 - Bad Request
```json
{
  "status": "error",
  "message": "Campo obrigatório ausente: api-key"
}
```

### 401 - Unauthorized
```json
{
  "status": "error",
  "message": "API key inválida ou inativa"
}
```

### 404 - Not Found
```json
{
  "status": "error",
  "message": "Pedido não encontrado"
}
```

### 405 - Method Not Allowed
```json
{
  "status": "error",
  "message": "Método não permitido. Use POST para pagamentos ou GET para consultas."
}
```

### 500 - Internal Server Error
```json
{
  "status": "error",
  "message": "Erro de conexão com banco de dados"
}
```

---

## 💻 Exemplos de Implementação

### JavaScript (Fetch API)
```javascript
// Processar Pagamento
async function processarPagamento(dados) {
  try {
    const response = await fetch('https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(dados)
    });
    
    const result = await response.json();
    console.log('Resultado:', result);
    return result;
  } catch (error) {
    console.error('Erro:', error);
  }
}

// Consultar Status
async function consultarStatus(apiKey, idTransaction) {
  try {
    const url = `https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php?api-key=${apiKey}&idTransaction=${idTransaction}`;
    const response = await fetch(url);
    const result = await response.json();
    console.log('Status:', result);
    return result;
  } catch (error) {
    console.error('Erro:', error);
  }
}

// Exemplo de uso
const dadosPagamento = {
  "api-key": "7115d6b39791cca550f4e3cc",
  "name": "João Silva",
  "cpf": "123.456.789-00",
  "amount": 50.00,
  "tipoPagamento": "PIX",
  "email": "joao@email.com"
};

// Processar pagamento
processarPagamento(dadosPagamento);

// Consultar status
consultarStatus("7115d6b39791cca550f4e3cc", "pay_8y3fyzdkeros3307");
```

### PHP (cURL)
```php
<?php
// Processar Pagamento
function processarPagamento($dados) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php');
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($dados));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($response, true);
}

// Consultar Status
function consultarStatus($apiKey, $idTransaction) {
    $url = "https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php?api-key={$apiKey}&idTransaction={$idTransaction}";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    
    $response = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($response, true);
}

// Exemplo de uso
$dadosPagamento = [
    'api-key' => '7115d6b39791cca550f4e3cc',
    'name' => 'João Silva',
    'cpf' => '123.456.789-00',
    'amount' => 50.00,
    'tipoPagamento' => 'PIX',
    'email' => 'joao@email.com'
];

// Processar pagamento
$resultado = processarPagamento($dadosPagamento);
print_r($resultado);

// Consultar status
$status = consultarStatus('7115d6b39791cca550f4e3cc', 'pay_8y3fyzdkeros3307');
print_r($status);
?>
```

### Python (requests)
```python
import requests
import json

# Processar Pagamento
def processar_pagamento(dados):
    url = 'https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php'
    headers = {'Content-Type': 'application/json'}
    
    response = requests.post(url, json=dados, headers=headers)
    return response.json()

# Consultar Status
def consultar_status(api_key, id_transaction):
    url = f'https://api.getpaybr.com/v1/gateway/processa_pagamento_api.php'
    params = {
        'api-key': api_key,
        'idTransaction': id_transaction
    }
    
    response = requests.get(url, params=params)
    return response.json()

# Exemplo de uso
dados_pagamento = {
    "api-key": "7115d6b39791cca550f4e3cc",
    "name": "João Silva",
    "cpf": "123.456.789-00",
    "amount": 50.00,
    "tipoPagamento": "PIX",
    "email": "joao@email.com"
}

# Processar pagamento
resultado = processar_pagamento(dados_pagamento)
print("Resultado:", resultado)

# Consultar status
status = consultar_status("7115d6b39791cca550f4e3cc", "pay_8y3fyzdkeros3307")
print("Status:", status)
```

---

## 🔧 Configurações

### Headers CORS
A API inclui os seguintes headers CORS para permitir requisições de diferentes origens:
```
Access-Control-Allow-Origin: *
Access-Control-Allow-Methods: GET, POST, OPTIONS
Access-Control-Allow-Headers: Content-Type, Authorization
```

### Timeout
- **Recomendado**: 30 segundos para pagamentos
- **Consulta de status**: 10 segundos

---

## 📝 Notas Importantes

1. **API Key**: Sempre mantenha sua API key segura e não a exponha em código cliente
2. **Validação**: A API valida automaticamente CPF/CNPJ e formata valores
3. **Sincronização**: O status é sincronizado automaticamente com o gateway Asaas
4. **Logs**: Todas as transações são registradas no banco de dados
5. **Webhooks**: Configure webhooks para receber notificações de mudança de status

---

## 🆘 Suporte

Para dúvidas ou problemas:
- Verifique se sua API key está ativa
- Confirme se todos os campos obrigatórios estão preenchidos
- Verifique a conectividade com o banco de dados
- Consulte os logs de erro do servidor

---

## 📅 Versão da API
- **Versão atual**: 1.0
- **Última atualização**: Agosto 2025
- **Compatibilidade**: PHP 7.4+
