<?php
/**
 * Gateway de Pagamento PIX para WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Classe do Gateway PIX
 */
class WC_PIX_Gateway extends WC_Payment_Gateway {
    
    /**
     * Construtor
     */
    public function __construct() {
        $this->id = 'pix_gateway';
        $this->icon = '';
        $this->has_fields = false;
        $this->method_title = __('Gateway PIX', 'woocommerce-pix-gateway');
        $this->method_description = __('Aceite pagamentos PIX através do gateway integrado com Asaas.', 'woocommerce-pix-gateway');
        
        // Configurações
        $this->init_form_fields();
        $this->init_settings();
        
        // Propriedades
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->api_key = $this->get_option('api_key');
        $this->api_url = $this->get_option('api_url');
        $this->sandbox = $this->get_option('sandbox');
        $this->debug = $this->get_option('debug');
        
        // Hooks
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_api_wc_pix_gateway', array($this, 'webhook_handler'));
        add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));
        add_action('wp_ajax_wc_pix_gateway_create_payment', array($this, 'ajax_create_payment'));
        add_action('wp_ajax_nopriv_wc_pix_gateway_create_payment', array($this, 'ajax_create_payment'));
    }
    
    /**
     * Campos de configuração do admin
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Habilitar/Desabilitar', 'woocommerce-pix-gateway'),
                'type' => 'checkbox',
                'label' => __('Habilitar Gateway PIX', 'woocommerce-pix-gateway'),
                'default' => 'no'
            ),
            'title' => array(
                'title' => __('Título', 'woocommerce-pix-gateway'),
                'type' => 'text',
                'description' => __('Título que o cliente verá durante o checkout.', 'woocommerce-pix-gateway'),
                'default' => __('Pagamento PIX', 'woocommerce-pix-gateway'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Descrição', 'woocommerce-pix-gateway'),
                'type' => 'textarea',
                'description' => __('Descrição que o cliente verá durante o checkout.', 'woocommerce-pix-gateway'),
                'default' => __('Pague com PIX de forma rápida e segura.', 'woocommerce-pix-gateway'),
                'desc_tip' => true,
            ),
            'api_key' => array(
                'title' => __('API Key', 'woocommerce-pix-gateway'),
                'type' => 'text',
                'description' => __('Sua chave de API única fornecida pelo gateway.', 'woocommerce-pix-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'api_url' => array(
                'title' => __('URL da API', 'woocommerce-pix-gateway'),
                'type' => 'text',
                'description' => __('URL base da API do gateway.', 'woocommerce-pix-gateway'),
                'default' => 'https://api.getpaybr.com/v1/gateway/',
                'desc_tip' => true,
            ),
            'sandbox' => array(
                'title' => __('Modo Sandbox', 'woocommerce-pix-gateway'),
                'type' => 'checkbox',
                'label' => __('Habilitar modo de teste', 'woocommerce-pix-gateway'),
                'default' => 'yes',
                'description' => __('Use para testar pagamentos sem processar valores reais.', 'woocommerce-pix-gateway'),
            ),
            'debug' => array(
                'title' => __('Modo Debug', 'woocommerce-pix-gateway'),
                'type' => 'checkbox',
                'label' => __('Habilitar logs de debug', 'woocommerce-pix-gateway'),
                'default' => 'no',
                'description' => __('Registra informações detalhadas para debugging.', 'woocommerce-pix-gateway'),
            ),
            'expiration_time' => array(
                'title' => __('Tempo de Expiração (minutos)', 'woocommerce-pix-gateway'),
                'type' => 'number',
                'description' => __('Tempo em minutos para o PIX expirar.', 'woocommerce-pix-gateway'),
                'default' => 30,
                'desc_tip' => true,
            ),
        );
    }
    
    /**
     * Processar pagamento
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return array(
                'result' => 'failure',
                'redirect' => ''
            );
        }
        
        try {
            // Criar pagamento no gateway
            $payment_data = $this->create_payment($order);
            
            if ($payment_data && isset($payment_data['status']) && $payment_data['status'] === 'success') {
                // Salvar dados do pagamento no pedido
                $order->update_meta_data('_pix_payment_id', $payment_data['data']['idTransaction']);
                $order->update_meta_data('_pix_external_reference', $payment_data['data']['externalReference']);
                $order->update_meta_data('_pix_payment_code', $payment_data['data']['paymentCode']);
                $order->update_meta_data('_pix_payment_code_base64', $payment_data['data']['paymentCodeBase64']);
                $order->update_meta_data('_pix_expires_at', $payment_data['data']['expiresAt']);
                $order->update_meta_data('_pix_pix_key', $payment_data['data']['pixKey']);
                
                // Atualizar status do pedido
                $order->update_status('pending', __('Aguardando pagamento PIX.', 'woocommerce-pix-gateway'));
                $order->save();
                
                // Redirecionar para página de pagamento
                return array(
                    'result' => 'success',
                    'redirect' => $order->get_checkout_payment_url(true)
                );
            } else {
                throw new Exception(__('Erro ao criar pagamento PIX.', 'woocommerce-pix-gateway'));
            }
            
        } catch (Exception $e) {
            $this->log('Erro ao processar pagamento: ' . $e->getMessage());
            
            wc_add_notice(__('Erro ao processar pagamento PIX. Tente novamente.', 'woocommerce-pix-gateway'), 'error');
            
            return array(
                'result' => 'failure',
                'redirect' => ''
            );
        }
    }
    
    /**
     * Criar pagamento no gateway
     */
    private function create_payment($order) {
        // $api_url = rtrim($this->api_url, '/') . '/criar-pagamento.php';
        $api_url = "https://web.getpaybr.com//api.getpaybr.com/v1/adquirente/asaas/processa_pagamento.php";
        
        // Obter dados do cliente
        $billing_first_name = $order->get_billing_first_name();
        $billing_last_name = $order->get_billing_last_name();
        $billing_email = $order->get_billing_email();
        $billing_phone = $order->get_billing_phone();
        
        // Tentar obter CPF/CNPJ dos meta dados
        $cpf_cnpj = '';
        if ($order->get_meta('_billing_cpf')) {
            $cpf_cnpj = preg_replace('/[^0-9]/', '', $order->get_meta('_billing_cpf'));
        } elseif ($order->get_meta('_billing_cnpj')) {
            $cpf_cnpj = preg_replace('/[^0-9]/', '', $order->get_meta('_billing_cnpj'));
        } else {
            // CPF padrão para teste
            $cpf_cnpj = '00000000000';
        }
        
        // Limpar telefone
        $telefone = preg_replace('/[^0-9]/', '', $billing_phone ?: '11999999999');
        
        // Obter endereço
        $billing_address_1 = $order->get_billing_address_1();
        $billing_address_2 = $order->get_billing_address_2();
        $billing_city = $order->get_billing_city();
        $billing_state = $order->get_billing_state();
        $billing_postcode = $order->get_billing_postcode();
        
        $payment_data = array(
            'valor' => number_format($order->get_total(), 2, '.', ''),
            'descricao' => 'Pedido #' . $order->get_order_number() . ' - ' . get_bloginfo('name'),
            'expira_em' => intval($this->get_option('expiration_time', 30)),
            'nome_cliente' => trim($billing_first_name . ' ' . $billing_last_name),
            'email_cliente' => $billing_email,
            'cpf_cnpj' => $cpf_cnpj,
            'telefone' => $telefone,
            'endereco' => array(
                'cep' => $billing_postcode ?: '00000-000',
                'logradouro' => $billing_address_1 ?: 'Endereço não informado',
                'numero' => '1',
                'complemento' => $billing_address_2 ?: '',
                'bairro' => 'Centro',
                'cidade' => $billing_city ?: 'São Paulo',
                'estado' => $billing_state ?: 'SP'
            )
        );
        
        $this->log('Enviando requisição para: ' . $api_url);
        $this->log('Dados do pagamento: ' . print_r($payment_data, true));
        
        $response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->api_key,
                'Accept' => 'application/json'
            ),
            'body' => json_encode($payment_data),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            $this->log('Erro na requisição: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $this->log('Resposta do gateway - Status: ' . $status_code . ' - Body: ' . $body);
        
        if ($status_code === 200) {
            $data = json_decode($body, true);
            return $data;
        }
        
        return false;
    }
    
    /**
     * Página de recebimento
     */
    public function receipt_page($order_id) {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            echo '<p>' . __('Pedido não encontrado.', 'woocommerce-pix-gateway') . '</p>';
            return;
        }
        
        $payment_id = $order->get_meta('_pix_payment_id');
        $payment_code = $order->get_meta('_pix_payment_code');
        $payment_code_base64 = $order->get_meta('_pix_payment_code_base64');
        $expires_at = $order->get_meta('_pix_expires_at');
        $pix_key = $order->get_meta('_pix_pix_key');
        
        if (!$payment_id) {
            echo '<p>' . __('Dados do pagamento não encontrados.', 'woocommerce-pix-gateway') . '</p>';
            return;
        }
        
        include WC_PIX_GATEWAY_PLUGIN_PATH . 'templates/payment-page.php';
    }
    
    /**
     * Handler do webhook
     */
    public function webhook_handler() {
        $this->log('Webhook recebido: ' . print_r($_POST, true));
        
        if (!isset($_POST['idTransaction']) || !isset($_POST['status'])) {
            $this->log('Webhook inválido - dados ausentes');
            status_header(400);
            exit;
        }
        
        $transaction_id = sanitize_text_field($_POST['idTransaction']);
        $status = sanitize_text_field($_POST['status']);
        $external_reference = isset($_POST['externalReference']) ? sanitize_text_field($_POST['externalReference']) : '';
        
        // Buscar pedido pelo ID da transação
        $orders = wc_get_orders(array(
            'meta_key' => '_pix_payment_id',
            'meta_value' => $transaction_id,
            'limit' => 1
        ));
        
        if (empty($orders)) {
            $this->log('Pedido não encontrado para transação: ' . $transaction_id);
            status_header(404);
            exit;
        }
        
        $order = $orders[0];
        
        // Atualizar status do pedido
        switch ($status) {
            case 'PAID_OUT':
                $order->payment_complete($transaction_id);
                $order->add_order_note(__('Pagamento PIX confirmado via webhook.', 'woocommerce-pix-gateway'));
                break;
                
            case 'OVERDUE':
                $order->update_status('failed', __('Pagamento PIX vencido.', 'woocommerce-pix-gateway'));
                break;
                
            case 'CANCELLED':
                $order->update_status('cancelled', __('Pagamento PIX cancelado.', 'woocommerce-pix-gateway'));
                break;
                
            case 'REFUNDED':
                $order->update_status('refunded', __('Pagamento PIX estornado.', 'woocommerce-pix-gateway'));
                break;
                
            default:
                $this->log('Status não reconhecido: ' . $status);
        }
        
        $this->log('Webhook processado com sucesso para pedido: ' . $order->get_id());
        
        status_header(200);
        echo 'OK';
        exit;
    }
    
    /**
     * AJAX para criar pagamento
     */
    public function ajax_create_payment() {
        check_ajax_referer('wc_pix_gateway_nonce', 'nonce');
        
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        
        if (!$order_id) {
            wp_die(__('ID do pedido inválido.', 'woocommerce-pix-gateway'));
        }
        
        $order = wc_get_order($order_id);
        
        if (!$order) {
            wp_die(__('Pedido não encontrado.', 'woocommerce-pix-gateway'));
        }
        
        $payment_data = $this->create_payment($order);
        
        if ($payment_data && isset($payment_data['status']) && $payment_data['status'] === 'success') {
            wp_send_json_success($payment_data);
        } else {
            wp_send_json_error(__('Erro ao criar pagamento PIX.', 'woocommerce-pix-gateway'));
        }
    }
    
    /**
     * Log de debug
     */
    private function log($message) {
        if ($this->debug === 'yes') {
            $logger = wc_get_logger();
            $logger->info($message, array('source' => 'pix_gateway'));
        }
    }
}
