# 🚀 API Gateway - Integração Asaas

Esta API permite integrar facilmente funcionalidades de pagamento PIX usando o Asaas como adquirente em suas aplicações.

## 📋 Funcionalidades

- ✅ **Criar Pagamento PIX**: Gera pagamentos PIX com QR Code e código de pagamento
- ✅ **Consultar Status**: Verifica o status atual de transações
- ✅ **Integração com Asaas**: Usa o Asaas como adquirente principal
- ✅ **Sistema de Taxas**: Aplica taxas configuráveis por usuário
- ✅ **Webhook/Postback**: Notificações automáticas de mudança de status
- ✅ **Validação de API Key**: Sistema seguro de autenticação

## 🔧 Requisitos

- PHP 7.2 ou superior
- MySQL/MariaDB
- Extensão cURL habilitada
- Acesso ao banco de dados configurado

## 📁 Estrutura de Arquivos

```
api.getpaybr.com/v1/gateway/
├── criar-pagamento.php      # Endpoint para criar pagamentos
├── consultar-status.php      # Endpoint para consultar status
├── exemplo-integracao.html   # Exemplo prático de integração
├── conectarbanco.php         # Configuração do banco de dados
└── README.md                 # Este arquivo
```

## 🚀 Instalação

### 1. Configuração do Banco de Dados

Certifique-se de que as seguintes tabelas existem no seu banco:

- `users_key` - Para armazenar as chaves de API dos usuários
- `adquirentes` - Para configurar adquirentes ativos
- `ad_asaas` - Para configurações específicas do Asaas
- `solicitacoes` - Para armazenar as transações
- `users` - Para dados dos usuários e taxas

### 2. Configuração do Asaas

1. Acesse o painel administrativo
2. Configure o Asaas como adquirente ativo
3. Insira sua `secret_key` do Asaas na tabela `ad_asaas`
4. Configure a URL da API do Asaas

### 3. Configuração da API Key

1. Gere uma API key única para cada usuário
2. Insira na tabela `users_key` com status 'ativo'
3. Configure as taxas do usuário na tabela `users`

## 📖 Como Usar

### Endpoint: Criar Pagamento

**URL:** `POST /criar-pagamento.php`

**Parâmetros obrigatórios:**
```json
{
    "amount": 100.50,
    "client": {
        "name": "João Silva",
        "document": "12345678901",
        "email": "joao@email.com"
    },
    "api-key": "sua_api_key_aqui"
}
```

**Parâmetros opcionais:**
```json
{
    "client": {
        "telefone": "11999999999"
    },
    "postback": "https://sualoja.com/webhook"
}
```

**Resposta de sucesso:**
```json
{
    "status": "success",
    "message": "Pagamento PIX criado com sucesso",
    "data": {
        "idTransaction": "pay_123456789",
        "externalReference": "abc123def456ghi789",
        "paymentCode": "00020101021226790014br.gov.bcb.pix...",
        "paymentCodeBase64": "iVBORw0KGgoAAAANSUhEUgAA...",
        "amount": "100.50",
        "status": "WAITING_FOR_APPROVAL",
        "expiresAt": "2025-01-28",
        "pixKey": "chave-pix@asaas.com"
    }
}
```

### Endpoint: Consultar Status

**URL:** `GET /consultar-status.php`

**Parâmetros:**
- `api-key` (obrigatório): Sua chave de API
- `idTransaction` (opcional): ID da transação no Asaas
- `externalReference` (opcional): Referência externa da transação

**Exemplo:**
```
GET /consultar-status.php?api-key=sua_api_key&idTransaction=pay_123456789
```

**Resposta de sucesso:**
```json
{
    "status": "success",
    "message": "Status consultado com sucesso",
    "data": {
        "idTransaction": "pay_123456789",
        "externalReference": "abc123def456ghi789",
        "amount": "100.50",
        "status": "PAID_OUT",
        "client": {
            "name": "João Silva",
            "document": "12345678901",
            "email": "joao@email.com",
            "telefone": "11999999999"
        },
        "createdAt": "2025-01-28 10:30:00",
        "adquirente": "asaas",
        "taxa_cash_in": "5.00",
        "deposito_liquido": "95.48",
        "asaas_status": "RECEIVED"
    }
}
```

## 🔐 Autenticação

Todas as requisições devem incluir uma `api-key` válida. Esta chave:

- É única para cada usuário
- Deve estar com status 'ativo' na tabela `users_key`
- É usada para identificar o usuário e aplicar suas taxas configuradas

## 💰 Sistema de Taxas

O sistema aplica automaticamente:

1. **Taxa do usuário**: Configurada na tabela `users.taxa_cash_in`
2. **Taxa do Asaas**: Configurada na tabela `ad_asaas.taxa_cash_in`
3. **Taxa fixa**: Configurada na tabela `app.taxa_pix_valor_real_cash_in_padrao`

O valor líquido é calculado: `valor_bruto - (valor_bruto * taxa_percentual) - taxa_fixa`

## 🔔 Webhook/Postback

Se configurar uma URL de postback, receberá notificações quando o status mudar:

```json
{
    "amount": "100.50",
    "idTransaction": "pay_123456789",
    "externalReference": "abc123def456ghi789",
    "paymentMethod": "PIX",
    "status": "PAID_OUT",
    "timestamp": "2025-01-28T10:30:00Z"
}
```

## 📊 Códigos de Status

| Status | Descrição | Significado |
|--------|-----------|-------------|
| `WAITING_FOR_APPROVAL` | Aguardando Aprovação | Pagamento criado, aguardando confirmação |
| `PAID_OUT` | Pago | Pagamento confirmado e processado |
| `OVERDUE` | Vencido | Pagamento não foi realizado no prazo |
| `CANCELLED` | Cancelado | Pagamento foi cancelado |
| `REFUNDED` | Estornado | Pagamento foi estornado |

## 🚨 Tratamento de Erros

### Códigos HTTP

- **200 OK**: Requisição processada com sucesso
- **400 Bad Request**: Dados da requisição inválidos ou incompletos
- **401 Unauthorized**: API key inválida ou inativa
- **404 Not Found**: Transação não encontrada
- **500 Internal Server Error**: Erro interno do servidor

### Exemplos de Erro

```json
{
    "status": "error",
    "message": "API key inválida ou inativa"
}
```

## 💡 Exemplos de Integração

### JavaScript (Fetch API)

```javascript
// Criar pagamento
const criarPagamento = async () => {
    const response = await fetch('/criar-pagamento.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            amount: 100.50,
            client: {
                name: "João Silva",
                document: "12345678901",
                email: "joao@email.com",
                telefone: "11999999999"
            },
            "api-key": "sua_api_key_aqui",
            postback: "https://sualoja.com/webhook"
        })
    });
    
    const data = await response.json();
    console.log(data);
};

// Consultar status
const consultarStatus = async (idTransaction) => {
    const response = await fetch(`/consultar-status.php?api-key=sua_api_key&idTransaction=${idTransaction}`);
    const data = await response.json();
    console.log(data);
};
```

### PHP (cURL)

```php
// Criar pagamento
$data = [
    'amount' => 100.50,
    'client' => [
        'name' => 'João Silva',
        'document' => '12345678901',
        'email' => 'joao@email.com',
        'telefone' => '11999999999'
    ],
    'api-key' => 'sua_api_key_aqui',
    'postback' => 'https://sualoja.com/webhook'
];

$ch = curl_init('/criar-pagamento.php');
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);

$response = curl_exec($ch);
curl_close($ch);

$result = json_decode($response, true);
print_r($result);
```

## 🧪 Testando a API

Use o arquivo `exemplo-integracao.html` para testar a API:

1. Abra o arquivo no navegador
2. Configure sua API key
3. Preencha os dados do cliente
4. Teste criar pagamentos e consultar status

## 🔧 Configurações Avançadas

### Personalizar Taxas

Edite as tabelas:
- `users.taxa_cash_in` - Taxa percentual por usuário
- `ad_asaas.taxa_cash_in` - Taxa do Asaas
- `app.taxa_pix_valor_real_cash_in_padrao` - Taxa fixa global

### Configurar Novos Adquirentes

1. Adicione na tabela `adquirentes`
2. Crie tabela específica (ex: `ad_novoadquirente`)
3. Implemente lógica no endpoint

## 🆘 Suporte

Para suporte técnico:

1. Verifique os logs de erro do PHP
2. Confirme as configurações do banco de dados
3. Valide as credenciais do Asaas
4. Entre em contato através do painel administrativo

## 📝 Changelog

### v1.0.0 (2025-01-28)
- ✅ Implementação inicial da API
- ✅ Endpoint para criar pagamentos PIX
- ✅ Endpoint para consultar status
- ✅ Integração com Asaas
- ✅ Sistema de taxas configurável
- ✅ Suporte a webhook/postback
- ✅ Documentação completa
- ✅ Exemplo de integração

## 📄 Licença

Este projeto é proprietário e confidencial.
