<?php
require_once 'config.php';

// Função asaasRequest será definida localmente no final do arquivo

header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *"); 
header("Access-Control-Allow-Methods: POST, OPTIONS"); 
header("Access-Control-Allow-Headers: Content-Type, Authorization"); 

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200); 
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $requestData = json_decode(file_get_contents('php://input'), true);

    // Validação dos dados obrigatórios
    if (!isset($requestData['amount'], $requestData['client'], $requestData['api-key'])) {
        http_response_code(400); 
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('MISSING_REQUIRED_FIELDS')
        ]);
        exit;
    }

    // Validação da estrutura do cliente
    if (!isset($requestData['client']['name'], $requestData['client']['document'], $requestData['client']['email'])) {
        http_response_code(400); 
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('INVALID_CLIENT_DATA')
        ]);
        exit;
    }

    include 'conectarbanco.php';
    $conn = new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);

    if ($conn->connect_error) {
        http_response_code(500);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('DATABASE_ERROR')
        ]);
        exit;
    }

    // Validar API key
    $sql = "SELECT user_id FROM users_key WHERE api_key = ? AND status = 'ativo'";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $requestData['api-key']);
    $stmt->execute();
    $stmt->store_result();

    if ($stmt->num_rows === 0) {
        http_response_code(401);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('INVALID_API_KEY')
        ]);
        $stmt->close();
        $conn->close();
        exit;
    }

    $stmt->bind_result($usuario);
    $stmt->fetch();
    $stmt->close();

    // Verificar se Asaas está ativo como adquirente
    $sql_adquirente = "SELECT adquirente FROM adquirentes WHERE adquirente = 'asaas' AND status = 1 LIMIT 1";
    $stmt_adquirente = $conn->prepare($sql_adquirente);
    $stmt_adquirente->execute();
    $stmt_adquirente->store_result();

    if ($stmt_adquirente->num_rows === 0) {
        http_response_code(503);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('ASAAS_UNAVAILABLE')
        ]);
        $stmt_adquirente->close();
        $conn->close();
        exit;
    }
    $stmt_adquirente->close();

    // Buscar configurações do Asaas
    $sql_asaas = "SELECT secret_key, url FROM ad_asaas WHERE id = 1";
    $stmt_asaas = $conn->prepare($sql_asaas);
    $stmt_asaas->execute();
    $stmt_asaas->bind_result($secret_key, $url);
    $stmt_asaas->fetch();
    $stmt_asaas->close();

    if (empty($secret_key)) {
        http_response_code(500);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('ASAAS_ERROR')
        ]);
        $conn->close();
        exit;
    }

    try {
        // Criar cliente no Asaas
        $customerData = [
            'name' => $requestData['client']['name'],
            'cpfCnpj' => preg_replace('/[^0-9]/', '', $requestData['client']['document']),
            'email' => $requestData['client']['email']
        ];

        if (isset($requestData['client']['telefone'])) {
            $customerData['phone'] = preg_replace('/[^0-9]/', '', $requestData['client']['telefone']);
        }
        var_dump($requestData);
        $customerResponse = asaasRequest('customers', 'POST', $customerData);
        var_dump($customerResponse);
        if (empty($customerResponse['id']) || isset($customerResponse['error'])) {
            $errorMessage = isset($customerResponse['error']) ? $customerResponse['message'] : ($customerResponse['message'] ?? 'Erro desconhecido');
            throw new Exception('Erro ao criar cliente no Asaas: ' . $errorMessage);
        }

        // Criar pagamento PIX no Asaas
        $paymentData = [
            'customer' => $customerResponse['id'],
            'billingType' => 'PIX',
            'value' => floatval($requestData['amount']),
            'dueDate' => date('Y-m-d'),
            'description' => 'Pagamento via PIX - ' . $requestData['client']['name']
        ];

        $paymentResponse = asaasRequest('payments', 'POST', $paymentData);
        
        if (empty($paymentResponse['id']) || isset($paymentResponse['error'])) {
            $errorMessage = isset($paymentResponse['error']) ? $paymentResponse['message'] : ($paymentResponse['message'] ?? 'Erro desconhecido');
            throw new Exception('Erro ao criar pagamento no Asaas: ' . $errorMessage);
        }

        // Buscar taxas do usuário
        $sql_taxa = "SELECT taxa_cash_in FROM users WHERE user_id = ?";
        $stmt_taxa = $conn->prepare($sql_taxa);
        $stmt_taxa->bind_param("s", $usuario);
        $stmt_taxa->execute();
        $stmt_taxa->bind_result($taxa_cash_in);
        $stmt_taxa->fetch();
        $stmt_taxa->close();

        if ($taxa_cash_in === null) {
            $taxa_cash_in = getConfig('DEFAULT_TAXA_CASH_IN', 5.00);
        }

        // Calcular valor líquido
        $taxa_cash_in_percentage = $taxa_cash_in / 100;
        $amount = floatval($requestData['amount']);
        $deposito_liquido = $amount - ($amount * $taxa_cash_in_percentage);

        // Gerar referência externa única
        $externalReference = generateRandomString(32);
        $status = 'WAITING_FOR_APPROVAL';
        $horaAtual = date('Y-m-d H:i:s');

        // Salvar na base de dados
        $sql_insert = "INSERT INTO solicitacoes 
        (user_id, externalreference, amount, client_name, client_document, client_email, real_data, paymentCode, idtransaction, paymentCodeBase64, status, adquirente_ref, taxa_cash_in, deposito_liquido, client_telefone, postback) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $stmt_insert = $conn->prepare($sql_insert);
        $postbackUrl = $requestData['postback'] ?? '';
        
        $stmt_insert->bind_param(
            "ssssssssssssssss",
            $usuario,
            $externalReference,
            $requestData['amount'],
            $requestData['client']['name'],
            $requestData['client']['document'],
            $requestData['client']['email'],
            $horaAtual,
            $paymentResponse['pix']['payload'] ?? '',
            $paymentResponse['id'],
            $paymentResponse['pix']['encodedImage'] ?? '',
            $status,
            'asaas',
            $taxa_cash_in,
            $deposito_liquido,
            $requestData['client']['telefone'] ?? '',
            $postbackUrl
        );

        if (!$stmt_insert->execute()) {
            throw new Exception('Erro ao salvar transação no banco de dados');
        }

        $stmt_insert->close();

        // Resposta de sucesso
        echo json_encode([
            "status" => "success",
            "message" => "Pagamento PIX criado com sucesso",
            "data" => [
                "idTransaction" => $paymentResponse['id'],
                "externalReference" => $externalReference,
                "paymentCode" => $paymentResponse['pix']['payload'] ?? '',
                "paymentCodeBase64" => $paymentResponse['pix']['encodedImage'] ?? '',
                "amount" => $requestData['amount'],
                "status" => $status,
                "expiresAt" => $paymentResponse['dueDate'] ?? '',
                "pixKey" => $paymentResponse['pix']['key'] ?? ''
            ]
        ]);

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            "status" => "error",
            "message" => $e->getMessage()
        ]);
    }

    $conn->close();
    } else {
        http_response_code(405);
        echo json_encode([
            "status" => "error", 
            "message" => getErrorMessage('INVALID_REQUEST_METHOD')
        ]);
    }

// Função para gerar string aleatória
function generateRandomString($length = 32) {
    $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $charLength = strlen($characters);
    $randomString = '';
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[random_int(0, $charLength - 1)];
    }
    return $randomString;
}

// Função para fazer requisições para o Asaas
function asaasRequest($endpoint, $method, $data = null) {
    // Buscar configurações do Asaas do banco de dados
    global $conn;
    
    $sql_asaas = "SELECT secret_key, url FROM ad_asaas WHERE id = 1";
    $stmt_asaas = $conn->prepare($sql_asaas);
    $stmt_asaas->execute();
    $stmt_asaas->bind_result($secret_key, $url);
    $stmt_asaas->fetch();
    $stmt_asaas->close();
    
    if (empty($secret_key)) {
        return [
            'error' => true,
            'message' => 'Secret key do Asaas não configurada',
            'http_code' => 500
        ];
    }
    
    // Construir URL completa
    $fullUrl = rtrim($url, '/') . '/' . ltrim($endpoint, '/');
    
    $headers = [
        'Content-Type: application/json',
        'access_token: ' . $secret_key,
        'User-Agent: Gateway-API/1.0'
    ];

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $fullUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST => $method,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_TIMEOUT => 30
    ]);
    
    if ($data && $method === 'POST') {
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    
    if (curl_errno($ch)) {
        curl_close($ch);
        return [
            'error' => true,
            'message' => 'Erro cURL: ' . curl_error($ch),
            'http_code' => 500
        ];
    }
    
    curl_close($ch);
    
    // Log da requisição para debug
    $logMessage = date('Y-m-d H:i:s') . " - REQUEST $method $fullUrl";
    if ($data) {
        $logMessage .= " body=" . json_encode($data);
    }
    $logMessage .= "\n";
    
    $logMessage .= date('Y-m-d H:i:s') . " - RESPONSE HTTP $httpCode body=" . $response . "\n";
    
    // Tentar criar o diretório de log se não existir
    $logDir = __DIR__ . '/../adquirente/asaas';
    if (is_dir($logDir) && is_writable($logDir)) {
        file_put_contents($logDir . '/asaas_http.log', $logMessage, FILE_APPEND);
    }
    
    $responseData = json_decode($response, true);
    
    // Verificar se houve erro HTTP
    if ($httpCode >= 400) {
        return [
            'error' => true,
            'http_code' => $httpCode,
            'message' => $responseData['message'] ?? 'Erro HTTP ' . $httpCode,
            'response' => $responseData
        ];
    }
    
    return $responseData;
}
?>
