<?php
/**
 * Configuração da API Gateway
 * 
 * Este arquivo contém as configurações principais da API
 * Ajuste conforme necessário para seu ambiente
 */

// Configurações do Banco de Dados
define('DB_HOST', 'localhost');
define('DB_USER', 'getpay_1'); // Altere para seu usuário do banco
define('DB_PASS', '984380500Oi'); // Altere para sua senha do banco
define('DB_NAME', 'getpay_1'); // Altere para o nome do seu banco

// Configurações da API
define('API_VERSION', 'v1.0.0');
define('API_NAME', 'Gateway API - Integração Asaas');
define('API_DESCRIPTION', 'API para integração de pagamentos PIX usando Asaas como adquirente');

// Configurações de Segurança
define('API_KEY_MIN_LENGTH', 32);
define('API_KEY_MAX_LENGTH', 64);
define('MAX_REQUESTS_PER_MINUTE', 100); // Rate limiting
define('REQUEST_TIMEOUT', 30); // Timeout em segundos

// Configurações do Asaas
define('ASAAS_SANDBOX_URL', 'https://sandbox.asaas.com/api/v3/');
define('ASAAS_PRODUCTION_URL', 'https://asaas.com/api/v3/');
define('ASAAS_TIMEOUT', 30);

// Configurações de Log
define('LOG_ENABLED', true);
define('LOG_LEVEL', 'INFO'); // DEBUG, INFO, WARNING, ERROR
define('LOG_FILE', __DIR__ . '/logs/api.log');

// Configurações de Cache
define('CACHE_ENABLED', false);
define('CACHE_TTL', 300); // 5 minutos

// Configurações de Webhook
define('WEBHOOK_TIMEOUT', 10);
define('WEBHOOK_RETRY_ATTEMPTS', 3);
define('WEBHOOK_RETRY_DELAY', 5); // segundos

// Configurações de Taxas Padrão
define('DEFAULT_TAXA_CASH_IN', 5.00); // 5%
define('DEFAULT_TAXA_CASH_OUT', 2.00); // 2%
define('DEFAULT_TAXA_FIXA', 0.00); // R$ 0,00

// Configurações de Validação
define('MIN_AMOUNT', 0.01); // Valor mínimo para pagamento
define('MAX_AMOUNT', 10000.00); // Valor máximo para pagamento
define('PIX_EXPIRATION_HOURS', 24); // Horas para expiração do PIX

// Configurações de Ambiente
define('ENVIRONMENT', 'production'); // development, staging, production
define('DEBUG_MODE', false);

// URLs da API
define('API_BASE_URL', 'https://api.getpaybr.com/v1/gateway/');
define('CREATE_PAYMENT_URL', API_BASE_URL . 'criar-pagamento.php');
define('CHECK_STATUS_URL', API_BASE_URL . 'consultar-status.php');

// Headers padrão
define('DEFAULT_HEADERS', [
    'Content-Type: application/json',
    'Accept: application/json',
    'User-Agent: Gateway-API/' . API_VERSION
]);

// Status de transação mapeados
define('TRANSACTION_STATUS_MAP', [
    'PENDING' => 'WAITING_FOR_APPROVAL',
    'RECEIVED' => 'PAID_OUT',
    'CONFIRMED' => 'PAID_OUT',
    'OVERDUE' => 'OVERDUE',
    'REFUNDED' => 'REFUNDED',
    'CANCELLED' => 'CANCELLED',
    'RECEIVED_IN_CASH' => 'PAID_OUT',
    'REFUND_REQUESTED' => 'REFUND_REQUESTED',
    'CHARGEBACK_REQUESTED' => 'CHARGEBACK_REQUESTED',
    'CHARGEBACK_DISPUTE' => 'CHARGEBACK_DISPUTE',
    'AWAITING_CHARGEBACK_REVERSAL' => 'AWAITING_CHARGEBACK_REVERSAL',
    'DUNNING_REQUESTED' => 'DUNNING_REQUESTED',
    'DUNNING_RECEIVED' => 'DUNNING_RECEIVED'
]);

// Mensagens de erro padrão
define('ERROR_MESSAGES', [
    'INVALID_API_KEY' => 'API key inválida ou inativa',
    'MISSING_REQUIRED_FIELDS' => 'Campos obrigatórios ausentes',
    'INVALID_AMOUNT' => 'Valor do pagamento inválido',
    'INVALID_CLIENT_DATA' => 'Dados do cliente inválidos',
    'ASAAS_UNAVAILABLE' => 'Adquirente Asaas não está disponível',
    'TRANSACTION_NOT_FOUND' => 'Transação não encontrada',
    'DATABASE_ERROR' => 'Erro de conexão com o banco de dados',
    'ASAAS_ERROR' => 'Erro na API do Asaas',
    'INVALID_REQUEST_METHOD' => 'Método de requisição não permitido'
]);

// Configurações de notificação
define('NOTIFICATION_EMAIL', 'suporte@gateway.com');
define('NOTIFICATION_SMS_ENABLED', false);
define('NOTIFICATION_SMS_URL', '');

// Configurações de backup
define('BACKUP_ENABLED', false);
define('BACKUP_INTERVAL', 'daily'); // daily, weekly, monthly
define('BACKUP_RETENTION_DAYS', 30);

// Função para obter configuração do ambiente
function getConfig($key, $default = null) {
    if (defined($key)) {
        return constant($key);
    }
    return $default;
}

// Função para verificar se está em modo de desenvolvimento
function isDevelopment() {
    return getConfig('ENVIRONMENT') === 'development' || getConfig('DEBUG_MODE') === true;
}

// Função para obter URL do Asaas baseada no ambiente
function getAsaasUrl() {
    if (isDevelopment()) {
        return getConfig('ASAAS_SANDBOX_URL');
    }
    return getConfig('ASAAS_PRODUCTION_URL');
}

// Função para log
function logMessage($level, $message, $context = []) {
    if (!getConfig('LOG_ENABLED')) {
        return;
    }
    
    $logLevel = getConfig('LOG_LEVEL');
    $allowedLevels = ['DEBUG', 'INFO', 'WARNING', 'ERROR'];
    
    if (array_search($level, $allowedLevels) < array_search($logLevel, $allowedLevels)) {
        return;
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] [{$level}] {$message}";
    
    if (!empty($context)) {
        $logEntry .= ' ' . json_encode($context);
    }
    
    $logEntry .= PHP_EOL;
    
    $logFile = getConfig('LOG_FILE');
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

// Função para validar API key
function validateApiKey($apiKey) {
    if (empty($apiKey)) {
        return false;
    }
    
    $minLength = getConfig('API_KEY_MIN_LENGTH');
    $maxLength = getConfig('API_KEY_MAX_LENGTH');
    
    if (strlen($apiKey) < $minLength || strlen($apiKey) > $maxLength) {
        return false;
    }
    
    // Validar formato (apenas caracteres alfanuméricos e hífens)
    if (!preg_match('/^[a-zA-Z0-9\-]+$/', $apiKey)) {
        return false;
    }
    
    return true;
}

// Função para validar valor do pagamento
function validateAmount($amount) {
    $minAmount = getConfig('MIN_AMOUNT');
    $maxAmount = getConfig('MAX_AMOUNT');
    
    if (!is_numeric($amount) || $amount < $minAmount || $amount > $maxAmount) {
        return false;
    }
    
    return true;
}

// Função para obter mensagem de erro
function getErrorMessage($key, $default = 'Erro desconhecido') {
    $messages = getConfig('ERROR_MESSAGES');
    return $messages[$key] ?? $default;
}

// Configurações de timezone
date_default_timezone_set('America/Sao_Paulo');

// Configurações de erro (apenas em desenvolvimento)
if (isDevelopment()) {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
} else {
    ini_set('display_errors', 0);
    ini_set('display_startup_errors', 0);
    error_reporting(0);
}

// Log de inicialização
logMessage('INFO', 'API Gateway inicializada', [
    'version' => API_VERSION,
    'environment' => getConfig('ENVIRONMENT'),
    'timestamp' => date('Y-m-d H:i:s')
]);
?>
