<?php
date_default_timezone_set('America/Sao_Paulo');
// Webhook de validação de saque (Asaas)
// Fluxo: recebe payload -> verifica pendência correspondente -> atualiza DB -> responde APPROVED/REFUSED

header('Content-Type: application/json; charset=utf-8');

$logFile = __DIR__ . '/asaas_webhook_validate_log.txt';
function log_webhook($message) {
    global $logFile;
    if (is_array($message) || is_object($message)) {
        $message = print_r($message, true);
    }
    @file_put_contents($logFile, "\n------\n[" . date('Y-m-d H:i:s') . "] " . $message . "\n", FILE_APPEND);
}

if (strtoupper($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    log_webhook('Método inválido: ' . ($_SERVER['REQUEST_METHOD'] ?? '')); 
    http_response_code(405);
    echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Método inválido']);
    exit;
}

$rawBody = file_get_contents('php://input');
log_webhook('BODY: ' . $rawBody);

$payload = json_decode($rawBody, true);
if (json_last_error() !== JSON_ERROR_NONE || empty($payload)) {
    log_webhook('JSON inválido');
    echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Payload inválido']);
    exit;
}

$type = $payload['type'] ?? null;

// Extração de dados principais independente do tipo
$pixKey = null;
$value = null;
$transferId = null;
$authorized = null;

if ($type === 'TRANSFER' && isset($payload['transfer'])) {
    $transfer = $payload['transfer'];
    $value = $transfer['value'] ?? null;
    $transferId = $transfer['id'] ?? null;
    $authorized = $transfer['authorized'] ?? null; // boolean
    if (!empty($transfer['bankAccount']['pixAddressKey'])) {
        $pixKey = $transfer['bankAccount']['pixAddressKey'];
    }
} elseif ($type === 'PIX_QR_CODE' && isset($payload['pixQrCode'])) {
    // Suporte básico para validação via PIX_QR_CODE, se necessário no futuro
    $pixQr = $payload['pixQrCode'];
    $value = $pixQr['value'] ?? null;
    $transferId = $pixQr['id'] ?? null;
    $authorized = ($pixQr['status'] ?? '') !== 'AWAITING_REQUEST' ? true : true; // tratar como autorizável
    if (!empty($pixQr['externalAccount']['addressKey'])) {
        $pixKey = $pixQr['externalAccount']['addressKey'];
    }
}

if (empty($transferId) && $value === null) {
    log_webhook('Payload sem transferId e sem valor.');
    echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Dados insuficientes']);
    exit;
}

// Normaliza valor para duas casas (compatível com DECIMAL(10,2))
$valueFloat = (float)$value;
$valueNormalized = number_format($valueFloat, 2, '.', '');

// Carrega conexão de banco
require_once __DIR__ . '/conectarbanco.php';

$mysqli = @new mysqli('localhost', $config['db_user'], $config['db_pass'], $config['db_name']);
if ($mysqli->connect_error) {
    log_webhook('Erro DB: ' . $mysqli->connect_error);
    echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Falha ao conectar ao banco']);
    exit;
}

// Localiza retirada prioritariamente pelo ID da transferência (referencia)
$retiradaRow = null;
if (!empty($transferId)) {
    $stmt = $mysqli->prepare("SELECT id, user_id, status FROM retiradas WHERE referencia = ? ORDER BY id DESC LIMIT 1");
    $stmt->bind_param('s', $transferId);
    if (!$stmt->execute()) {
        log_webhook('Falha ao consultar retirada por referencia: ' . $stmt->error);
        echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Erro ao consultar retirada']);
        exit;
    }
    $result = $stmt->get_result();
    if ($result && $result->num_rows > 0) {
        $retiradaRow = $result->fetch_assoc();
    }
}

if ($retiradaRow === null) {
    log_webhook('Retirada não encontrada por referencia transferId=' . ($transferId ?? 'null'));
    echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Transferência não encontrada']);
    exit;
}

$row = $retiradaRow;
$retiradaId = (int)$row['id'];
$userId = $row['user_id'];

// Idempotência: se já foi aprovada por algum motivo, apenas aprovar novamente
if ((string)$row['status'] === '1') {
    log_webhook('Retirada já aprovada. id=' . $retiradaId);
    echo json_encode(['status' => 'APPROVED']);
    exit;
}

// Opcional: aqui daria para validar saldo do usuário antes de aprovar.
// Sem o schema exato, vamos seguir aprovando a solicitação que está pendente.

// Marca como aprovado no banco
$dataAgora = date('Y-m-d H:i:s');
$update = $mysqli->prepare("UPDATE retiradas SET status = '1', data_pagamento = ? WHERE id = ?");
$update->bind_param('si', $dataAgora, $retiradaId);
if (!$update->execute()) {
    log_webhook('Falha ao atualizar retirada id=' . $retiradaId . ' err=' . $update->error);
    echo json_encode(['status' => 'REFUSED', 'refuseReason' => 'Erro ao atualizar retirada']);
    exit;
}

log_webhook('Retirada aprovada id=' . $retiradaId . ' user=' . $userId . ' valor=' . $valueNormalized . ' transferId=' . ($transferId ?? ''));

echo json_encode(['status' => 'APPROVED']);
exit;
?>